;--------------------------------
; General Installer Settings
;--------------------------------
!include StrFunc.nsh
${StrStr}
${StrTok}

!define APPNAME "ogamma Visual Logger for OPC"
!define COMPANYNAME "One-Way Automation Inc."
!define COMPANY "opcfy"
!define VERSION "4.2.0"
!define PROGRAMDATA "C:\ProgramData"

!define OVL_SERVICE_NAME "One-Way Automation Visual Logger"
!define HA_SERVICE_NAME  "One-Way Automation HA Node Manager"

; Publisher name shown in the installer GUI
!define MUI_WELCOMEPAGE_TITLE "${APPNAME} Setup"
!define MUI_WELCOMEPAGE_TEXT "Publisher: ${COMPANYNAME}"

OutFile "..\..\..\distr-packages\ogamma-logger-windows-installer-4.2.0.exe"
InstallDir "$PROGRAMFILES64\${COMPANY}\ovl"
InstallDirRegKey HKLM "Software\${COMPANY}\ovl" "Install_Dir"

Icon "..\..\..\runtime\public_html\pic\favicon.ico"
UninstallIcon "..\..\..\runtime\public_html\pic\favicon.ico"
Name "ogamma Visual Logger for OPC"
Caption "ogamma Visual Logger for OPC"


!define MUI_HEADERIMAGE
!define MUI_HEADERIMAGE_RIGHT
!define MUI_HEADERIMAGE_BITMAP "..\..\..\runtime\public_html\pic\1wa-logo.bmp"

Var DATA_DIR
Var UNINSTALL_REMOVE_DATA
Var UNINSTALL_CHECKBOX
Var INSTALL_HA_NODE
Var HA_CLUSTER_ID
Var HA_NODE_ID
Var HA_HEALTH_URL
Var HA_CHECKBOX
Var HA_CLUSTER_TEXT
Var HA_NODE_TEXT
Var HA_URL_TEXT
Var OVL_PORT_TEXT
Var OVL_PROTOCOL_TEXT
Var OVL_PORT
Var OVL_PROTOCOL
Var SavedPort
Var SavedProtocol
Var INFO_TEXT
VAR INFO_LINK
Var HOST_NAME
Var DATA_DIR_VAR
Var INSTDIR_VAR 

; Maintenance mode variables
Var MAINTENANCE_ACTION
Var RADIO_REPAIR
Var RADIO_UPGRADE
Var RADIO_CANCEL

;--------------------------------
; Pages
;--------------------------------
; TODO: uncomment to enable editing of the install folder 
; Page directory
Page custom MaintenancePage MaintenancePageLeave
Page custom DataDirPage DataDirPageLeave
Page custom LoggerConfigPage LoggerConfigPageLeave
Page custom HANodePage HANodePageLeave
Page instfiles
Page custom InfoPage

UninstPage custom un.DataPage un.DataPageLeave
UninstPage instfiles

!include "MUI2.nsh"
!include "nsDialogs.nsh"

;--------------------------------
; Functions
;--------------------------------

Function .onInit
  StrCpy $DATA_DIR "${PROGRAMDATA}\${COMPANY}\ovl"
    ; Get computer hostname
  System::Call 'kernel32::GetComputerNameW(w .r0, *i ${NSIS_MAX_STRLEN})'
  StrCmp $0 "" 0 +2
    StrCpy $0 "localhost"
  StrCpy $HOST_NAME $0
  
    ; Default values
  StrCpy $OVL_PORT "4880"
  StrCpy $OVL_PROTOCOL "http"
  
    ; Also initialize saved values
  StrCpy $SavedPort $OVL_PORT
  StrCpy $SavedProtocol $OVL_PROTOCOL

  ; Detect existing installation
  ReadRegStr $INSTDIR_VAR HKLM "Software\${COMPANY}\ovl" "Install_Dir"
  ReadRegStr $R0 HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${APPNAME}" "DisplayVersion"

  StrCmp $INSTDIR_VAR "" no_previous_install has_previous_install

  no_previous_install:
	StrCpy $MAINTENANCE_ACTION "fresh"
    Return

  has_previous_install:
    ; Force showing our nsDialogs maintenance page
    StrCpy $MAINTENANCE_ACTION ""  ; reset
	ReadRegStr $DATA_DIR_VAR HKLM "Software\${COMPANY}\ovl" "Data_Dir"
	ReadRegStr $DATA_DIR HKLM "Software\${COMPANY}\ovl" "Data_Dir"
FunctionEnd

Function MaintenancePage
  ${If} $MAINTENANCE_ACTION == "fresh"
    ; No previous install → skip this page
    Abort
  ${EndIf}
  
  nsDialogs::Create 1018
  Pop $0
  ${If} $0 == error
    Abort
  ${EndIf}

  ${If} $R0 == "${VERSION}"
    ; Same version → only Repair/Cancel
    ${NSD_CreateLabel} 0 0 100% 24u "Version ${VERSION} of ${APPNAME} is already installed. Choose an action:"

    ${NSD_CreateRadioButton} 0 30u 100% 12u "Repair existing installation"
    Pop $RADIO_REPAIR

    ${NSD_CreateRadioButton} 0 50u 100% 12u "Cancel installation"
    Pop $RADIO_CANCEL

    ; Default selection: Repair
    SendMessage $RADIO_REPAIR ${BM_SETCHECK} ${BST_CHECKED} 0
  ${Else}
    ; Different version → Repair/Upgrade/Cancel
    ${NSD_CreateLabel} 0 0 100% 24u "An existing installation of ${APPNAME} was detected (version $R0). Choose an action:"

    ${NSD_CreateRadioButton} 0 30u 100% 12u "Repair existing installation"
    Pop $RADIO_REPAIR

    ${NSD_CreateRadioButton} 0 50u 100% 12u "Change to version ${VERSION}"
    Pop $RADIO_UPGRADE

    ${NSD_CreateRadioButton} 0 70u 100% 12u "Cancel installation"
    Pop $RADIO_CANCEL

    ; Default selection: Upgrade
    SendMessage $RADIO_UPGRADE ${BM_SETCHECK} ${BST_CHECKED} 0
  ${EndIf}

  nsDialogs::Show
FunctionEnd

Function MaintenancePageLeave
  ${NSD_GetState} $RADIO_REPAIR $R1
  ${If} $R1 == 1
    StrCpy $MAINTENANCE_ACTION "repair"
    Return
  ${EndIf}

  ${NSD_GetState} $RADIO_UPGRADE $R1
  ${If} $R1 == 1
    StrCpy $MAINTENANCE_ACTION "upgrade"
    Return
  ${EndIf}

  ${NSD_GetState} $RADIO_CANCEL $R1
  ${If} $R1 == 1
    StrCpy $MAINTENANCE_ACTION "cancel"
    Quit
  ${EndIf}
FunctionEnd

; Data directory page
Function OnBrowseDataDir
  ; Popup the folder browser dialog with default starting directory
  nsDialogs::SelectFolderDialog "Select Data Folder" "$DATA_DIR"
  Pop $0 ; $0 holds the selected path
  
  ; If a selection was made (not cancelled), update the text field
  ${If} $0 != error
    ${NSD_SetText} $DATA_DIR_VAR $0
  ${EndIf}
FunctionEnd

Function DataDirPage
  nsDialogs::Create 1018
  Pop $0
  ${If} $0 == error
    Abort
  ${EndIf}

  ${NSD_CreateLabel} 0 0 100% 12u "Select folder for application data files:"
  Pop $1

  ; Create text field for data directory path
  ${NSD_CreateText} 0 14u 80% 12u "$DATA_DIR"
  Pop $DATA_DIR_VAR

  ; Create browse button
  ${NSD_CreateBrowseButton} 81% 14u 19% 12u "Browse..."
  Pop $2
  ${NSD_OnClick} $2 OnBrowseDataDir

  ; If not a fresh install, disable field and hide browse button
  ${If} $MAINTENANCE_ACTION != "fresh"
    EnableWindow $DATA_DIR_VAR 0  ; Disable the text field
    ShowWindow $2 0  ; Hide the browse button
    ${NSD_CreateLabel} 0 40u 100% 12u "Note: Data folder cannot be changed during upgrade/repair"
    Pop $3
    SetCtlColors $3 0xFF0000 0xFFFFFF  ; Red text on white background
  ${EndIf}
  
  nsDialogs::Show
FunctionEnd

Function DataDirPageLeave
  ${NSD_GetText} $DATA_DIR_VAR $DATA_DIR
FunctionEnd

Function LoggerConfigPage
  nsDialogs::Create 1018
  Pop $0
  ${If} $0 == error
    Abort
  ${EndIf}

  ; --- Initialize defaults if not set ---
  StrCmp $SavedProtocol "" 0 protocol_has_value
    StrCpy $SavedProtocol "http"
  protocol_has_value:

  StrCmp $SavedPort "" 0 port_has_value
    StrCpy $SavedPort "4880"
  port_has_value:

  ; --- Create labels and fields ---
  ; Protocol first
  ${NSD_CreateLabel} 0 0 100% 12u "Configuration Endpoint Protocol (http or https):"
  Pop $0

  ${NSD_CreateText} 0 15u 100% 12u $SavedProtocol
  Pop $OVL_PROTOCOL_TEXT

  ; Port second
  ${NSD_CreateLabel} 0 40u 100% 12u "Configuration Endpoint Port Number:"
  Pop $0

  ${NSD_CreateText} 0 55u 100% 12u $SavedPort
  Pop $OVL_PORT_TEXT

  nsDialogs::Show
FunctionEnd

Function LoggerConfigPageLeave
  ${NSD_GetText} $OVL_PORT_TEXT $OVL_PORT
  ${NSD_GetText} $OVL_PROTOCOL_TEXT $OVL_PROTOCOL

  ; Save values so they persist
  StrCpy $SavedPort $OVL_PORT
  StrCpy $SavedProtocol $OVL_PROTOCOL
  
  ; --- Validate port ---
  StrCmp $OVL_PORT "" 0 port_not_empty
    MessageBox MB_ICONSTOP "Port number cannot be empty."
    Abort
  port_not_empty:

  ; --- Validate protocol ---
  StrCmp $OVL_PROTOCOL "http" protocol_ok
  StrCmp $OVL_PROTOCOL "https" protocol_ok
    MessageBox MB_ICONSTOP "Protocol must be 'http' or 'https'."
    Abort
  protocol_ok:

; Show banner
  Banner::show "Checking if the $OVL_PROTOCOL port $OVL_PORT is free..."

; Run PowerShell check: exit 1 if INUSE, exit 0 if FREE
  nsExec::ExecToStack 'powershell -NoProfile -ExecutionPolicy Bypass -Command "try {Get-NetTCPConnection -State Listen -LocalPort $OVL_PORT -ErrorAction Stop | Out-Null; exit 1} catch {exit 0}"'
  Pop $R0   ; This is the exit code

  ; Evaluate result
  StrCmp $R0 "1" port_in_use port_free

  port_in_use:
    nsExec::ExecToStack 'powershell -NoProfile -ExecutionPolicy Bypass -Command "try {if (Get-NetTCPConnection -LocalPort $OVL_PORT -State Listen -OwningProcess (Get-Process -Name "ogamma-logger" -ErrorAction SilentlyContinue).Id -ErrorAction SilentlyContinue) { exit 2 }} catch {exit 1}"'
	Pop $R0 
	; Close banner
    Banner::destroy
    StrCmp $R0 "2" port_in_use_by_ovl port_in_use_by_other
	
	port_in_use_by_ovl:
      DetailPrint "Port $OVL_PORT is used by process ogamma-logger.exe - OK."
      Return
	port_in_use_by_other:
      MessageBox MB_ICONSTOP "Port $OVL_PORT is already in use. Please choose another port."
      Abort
  port_free:
    ; Close banner
    Banner::destroy
    DetailPrint "Port $OVL_PORT is free."
    Return
FunctionEnd

Function HANodePage
  nsDialogs::Create 1018
  Pop $0
  ${If} $0 == error
    Abort
  ${EndIf}

  ; Checkbox to install ha-node service
  ${NSD_CreateCheckbox} 0 0 100% 12u "Install HA (High Availability) Node service."
  ;  In this case, the instance of the ogamma Visual Logger for OPC will be a member node of the HA cluster."
  Pop $HA_CHECKBOX
  ${NSD_OnClick} $HA_CHECKBOX HANodeToggleFields
  ; TODO - add tooltips ${NSD_SetToolTip} $HA_CHECKBOX "If this option is turned ON, additional Windows Service ""ha-node"" will be installed, facilitating implementation of the High Avialibility feature: instance of the ogamma Visual Logger for OPC running in this host will be member node of the HA Cluster."
  
  ; Cluster ID field
  ${NSD_CreateLabel} 0 20u 100% 12u "HA Cluster ID:"
  Pop $1
  ${NSD_CreateText} 0 32u 100% 12u "ha-cluster"
  Pop $HA_CLUSTER_TEXT

  ; Node ID field
  ${NSD_CreateLabel} 0 52u 100% 12u "HA Node ID:"
  Pop $2
  ${NSD_CreateText} 0 64u 100% 12u "ha-node-1"
  Pop $HA_NODE_TEXT

  ; Health Check URL field
  ${NSD_CreateLabel} 0 84u 100% 12u "Application (OVL) Health Check URL:"
  Pop $3
  ${NSD_CreateText} 0 96u 100% 12u "$OVL_PROTOCOL://$HOST_NAME:$OVL_PORT/health"
  Pop $HA_URL_TEXT
  
  ; Initially disable fields
  EnableWindow $HA_CLUSTER_TEXT 0
  EnableWindow $HA_NODE_TEXT 0
  EnableWindow $HA_URL_TEXT 0
  
  nsDialogs::Show
FunctionEnd

Function HANodePageLeave
  ${NSD_GetState} $HA_CHECKBOX $INSTALL_HA_NODE
  ${NSD_GetText} $HA_CLUSTER_TEXT $HA_CLUSTER_ID
  ${NSD_GetText} $HA_NODE_TEXT $HA_NODE_ID
  ${NSD_GetText} $HA_URL_TEXT $HA_HEALTH_URL
FunctionEnd

Function HANodeToggleFields
  ${NSD_GetState} $HA_CHECKBOX $INSTALL_HA_NODE
  ${If} $INSTALL_HA_NODE == 1
    EnableWindow $HA_CLUSTER_TEXT 1
    EnableWindow $HA_NODE_TEXT 1
    EnableWindow $HA_URL_TEXT 1
  ${Else}
    EnableWindow $HA_CLUSTER_TEXT 0
    EnableWindow $HA_NODE_TEXT 0
    EnableWindow $HA_URL_TEXT 0
  ${EndIf}
FunctionEnd

; Uninstaller custom page
Function un.DataPage
  ReadRegStr $DATA_DIR_VAR HKLM "Software\${COMPANY}\ovl" "Data_Dir"
  nsDialogs::Create 1018
  Pop $0
  ${If} $0 == error
    Abort
  ${EndIf}

  ;${NSD_CreateLabel} 0 0 100% 12u "Do you also want to remove application data files (logs/configs) from ?"
  ;Pop $1

  ${NSD_CreateCheckbox} 0 14u 100% 12u "Remove data folder $DATA_DIR_VAR"
  Pop $UNINSTALL_CHECKBOX

  nsDialogs::Show
FunctionEnd

Function un.DataPageLeave
  ${NSD_GetState} $UNINSTALL_CHECKBOX $UNINSTALL_REMOVE_DATA
FunctionEnd

Function InfoPage
  nsDialogs::Create 1018
  Pop $0
  ${If} $0 == error
    Abort
  ${EndIf}

  ; Message
  ${NSD_CreateLabel} 0 0 100% 24u "Installation complete. Open Web configuration page at:"
  Pop $INFO_TEXT

  ; Clickable link
  ${NSD_CreateLink} 0 30u 100% 12u "$OVL_PROTOCOL://$HOST_NAME:$OVL_PORT"
  Pop $INFO_LINK

  ; Add click handler to open link
  ${NSD_OnClick} $INFO_LINK InfoPage_OpenLink

  nsDialogs::Show
FunctionEnd

Function InfoPage_OpenLink
  ; $INFO_LINK contains the control handle, so retrieve its text
  ${NSD_GetText} $INFO_LINK $1
  ExecShell "open" "$1"
FunctionEnd


;--------------------------------
; Install Sections
;--------------------------------
Section "Install dependency - Visual C / C++ runtime libraries" SEC01
  ; Check if VC++ 2019 Redistributable is installed
  ClearErrors
  ReadRegDWORD $0 HKLM "SOFTWARE\Microsoft\VisualStudio\14.0\VC\Runtimes\x64" "Installed"
  ${If} $0 != 1
    ExecWait '"$INSTDIR\vc_redist.x64.exe" /install /quiet /norestart'
  ${Else}
    DetailPrint "VC++ 2019 Redistributable already installed, skipping..."
  ${EndIf}
SectionEnd

Section "Install dependency: ODBC Driver 17 for SQL Server" SEC02
  ; Check if registry entry exists
  ReadRegStr $R0 HKLM "SOFTWARE\ODBC\ODBCINST.INI\ODBC Driver 17 for SQL Server" "Driver"

  StrCmp $R0 "" not_installed installed

  not_installed:
    DetailPrint "ODBC Driver 17 not detected. Installing..."
    ; Make sure the MSI is staged in $PLUGINSDIR
    File "msodbcsql.msi"
    ExecWait 'msiexec /i "$PLUGINSDIR\msodbcsql.msi" /qn /norestart /l*v "$PLUGINSDIR\odbc_install.log"' $R1
    StrCmp $R1 0 install_ok install_failed

    install_ok:
      DetailPrint "ODBC Driver 17 installed successfully."
      Goto done

    install_failed:
      MessageBox MB_ICONSTOP "Failed to install ODBC Driver 17. Error code: $R1. Try to install it manually using the file $INSTDIR\msodbcsql.msi"
      Abort

  installed:
    DetailPrint "ODBC Driver 17 already installed. Skipping installation."
    Goto done
  done:
SectionEnd


Function StopServices
  DetailPrint "Stopping service ${OVL_SERVICE_NAME} ..."
  ExecWait '"$INSTDIR_VAR\nssm.exe" stop "${OVL_SERVICE_NAME}"'
  DetailPrint "Stopping service ${HA_SERVICE_NAME} ..."
  ExecWait '"$INSTDIR_VAR\nssm.exe" stop "${HA_SERVICE_NAME}"'
FunctionEnd

Section "Install Application Files" SEC03
  ${If} $MAINTENANCE_ACTION == "repair"
	Call StopServices
    DetailPrint "Repairing existing installation..."
  ${ElseIf} $MAINTENANCE_ACTION == "upgrade"
	Call StopServices
    DetailPrint "Upgrading from version $R0 to ${VERSION}..."
  ${Else}
    DetailPrint "Fresh installation..."
  ${EndIf}
  
  ; Copy application files
  SetOutPath $INSTDIR
  File /r /x .vs *

  ; Include NSSM
  File "C:\nssm-2.24\win64\nssm.exe"

  ; Include ha_node.exe
  File "..\..\..\..\ha-node\build-windows\Release\ha_node.exe"

  ; Registry entries
  WriteRegStr HKLM "Software\${COMPANY}\ovl" "Install_Dir" "$INSTDIR"
  WriteRegStr HKLM "Software\${COMPANY}\ovl" "Data_Dir" "$DATA_DIR"
  
  WriteRegStr HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${APPNAME}" "Publisher" "${COMPANYNAME}"
  WriteRegStr HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${APPNAME}" "DisplayName" "${APPNAME}"
  WriteRegStr HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${APPNAME}" "DisplayVersion" "${VERSION}"
  WriteRegStr HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${APPNAME}" "UninstallString" "$INSTDIR\Uninstall.exe"

  ; Write uninstaller
  WriteUninstaller "$INSTDIR\Uninstall.exe"
  
  
SectionEnd

Function AddFirewallRule
    ; Add a port rule using the SimpleFC plugin
    ; SimpleFC::AddPort [port] [name] [protocol] [scope] [ip_version] [remote_addresses] [status]
    SimpleFC::AddPort "$OVL_PORT" "${APPNAME}" "6" "0" "2" "" "1"
    Pop $0
    ${If} $0 == 0
        DetailPrint "Firewall rule for port $OVL_PORT added successfully."
    ${Else}
        DetailPrint "Failed to add firewall rule. Error: $0"
    ${EndIf}
FunctionEnd


Section "Add Firewall Rule" SEC04
    ; Check if the port is already enabled
    SimpleFC::IsPortEnabled "$OVL_PORT" "6" ; 6 = TCP protocol
    Pop $0  ; Pops the first return value (error code)
    Pop $1  ; Pops the second return value (port status)

    ; Evaluate the result
    ${If} $0 == 0
        ${If} $1 == 1
            DetailPrint "Port $OVL_PORT (TCP) is already enabled. Skipping rule addition."
        ${Else}
            DetailPrint "Port $OVL_PORT (TCP) rule exists but is disabled. Enabling..."
            ; Option 1: Enable the existing rule
            SimpleFC::EnableDisablePort "$OVL_PORT" "6" "1"
            Pop $0
            ${If} $0 != 0
                DetailPrint "Failed to enable the existing port rule. Adding a new one."
                ; --- Call your function to add a new rule here ---
                Call AddFirewallRule
            ${EndIf}
        ${EndIf}
    ${Else}
        DetailPrint "Port $OVL_PORT (TCP) is not in the exception list. Adding a new rule."
        ; --- Call your function to add a new rule here ---
        Call AddFirewallRule
    ${EndIf}
SectionEnd

Section "Configure Windows Services" SEC05
  CreateDirectory "$DATA_DIR"
  CreateDirectory "$DATA_DIR\data"

  ; ogamma-logger service
  ExecWait '"$INSTDIR\nssm.exe" install "${OVL_SERVICE_NAME}" "$INSTDIR\ogamma-logger.exe"'
  ExecWait '"$INSTDIR\nssm.exe" set "${OVL_SERVICE_NAME}" Description "ogamma Visual Logger for OPC"'
  ExecWait '"$INSTDIR\nssm.exe" set "${OVL_SERVICE_NAME}" AppDirectory "$DATA_DIR"'
  ExecWait '"$INSTDIR\nssm.exe" set "${OVL_SERVICE_NAME}" AppStdout "$DATA_DIR\data\ovl-service-console-output.txt"'
  ExecWait '"$INSTDIR\nssm.exe" set "${OVL_SERVICE_NAME}" AppStderr "$DATA_DIR\data\ovl-service-console-output.txt"'
  ExecWait '"$INSTDIR\nssm.exe" set "${OVL_SERVICE_NAME}" AppEnvironmentExtra "OVL_PORT=$OVL_PORT" "OVL_PROTOCOL=$OVL_PROTOCOL"'
  
  ; ha-node service (optional)
  ${If} $INSTALL_HA_NODE == 1
	ExecWait '"$INSTDIR\nssm.exe" install "${HA_SERVICE_NAME}" "$INSTDIR\ha_node.exe"'
    ExecWait '"$INSTDIR\nssm.exe" set "${HA_SERVICE_NAME}" Description "High-Availability Node service for ogamma Visual Logger for OPC, cluster $HA_CLUSTER_ID, node $HA_NODE_ID"'
	ExecWait '"$INSTDIR\nssm.exe" set "${HA_SERVICE_NAME}" AppDirectory "$DATA_DIR"'
	ExecWait '"$INSTDIR\nssm.exe" set "${HA_SERVICE_NAME}" AppStdout "$DATA_DIR\data\ha-service-console-output.txt"'
	ExecWait '"$INSTDIR\nssm.exe" set "${HA_SERVICE_NAME}" AppStderr "$DATA_DIR\data\ha-service-console-output.txt"'  
	; Set environment variables for ha-node
    ExecWait '"$INSTDIR\nssm.exe" set "${HA_SERVICE_NAME}" AppEnvironmentExtra "HA_CLUSTER_ID=$HA_CLUSTER_ID" "HA_NODE_ID=$HA_NODE_ID" "HA_HEALTH_CHECK_URL=$HA_HEALTH_URL" "HA_HEALTH_CHECK_ENABLED=ON"'
    ExecWait '"$INSTDIR\nssm.exe" start "${HA_SERVICE_NAME}"'
  ${Else}
    DetailPrint "Skipping High Availability Node service installation"
  ${EndIf}
 
  ExecWait '"$INSTDIR\nssm.exe" start "${OVL_SERVICE_NAME}"'
  
SectionEnd


Section "Create Uninstall Shortcut" SEC06
  CreateDirectory "$SMPROGRAMS\${COMPANY}"
  CreateShortCut "$SMPROGRAMS\${COMPANY}\Uninstall ${APPNAME}.lnk" "$INSTDIR\Uninstall.exe" "" "$INSTDIR\Uninstall.exe" 0
SectionEnd

;--------------------------------
; Uninstaller Section
;--------------------------------
Section "Uninstall"
; Read the actual install directory from registry
  ReadRegStr $INSTDIR_VAR HKLM "Software\${COMPANY}\ovl" "Install_Dir"
  StrCmp $INSTDIR_VAR "" 0 +2
    StrCpy $INSTDIR_VAR $INSTDIR   ; fallback if registry missing

  DetailPrint "Stopping service ${OVL_SERVICE_NAME} ..."
  ExecWait '"$INSTDIR_VAR\nssm.exe" stop "${OVL_SERVICE_NAME}"'
  DetailPrint "Removing service ${OVL_SERVICE_NAME} ..."
  ExecWait '"$INSTDIR_VAR\nssm.exe" remove "${OVL_SERVICE_NAME}" confirm'

  DetailPrint "Stopping service ${HA_SERVICE_NAME} ..."
  ExecWait '"$INSTDIR_VAR\nssm.exe" stop "${HA_SERVICE_NAME}"'
  DetailPrint "Removing service ${HA_SERVICE_NAME} ..."
  ExecWait '"$INSTDIR_VAR\nssm.exe" remove "${HA_SERVICE_NAME}" confirm'

; Read the data folder path from registry
  ReadRegStr $DATA_DIR_VAR HKLM "Software\${COMPANY}\ovl" "Data_Dir"

  DetailPrint "Cleaning registry entries..."
  ; Remove uninstall registry entry
  DeleteRegKey HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${APPNAME}"

  ; Remove app registry entry
  DeleteRegKey HKLM "Software\${COMPANY}\ovl"
 
  DetailPrint "Removing Start Menu shortcuts..."
  Delete "$SMPROGRAMS\${COMPANY}\${APPNAME}.lnk"
  Delete "$SMPROGRAMS\${COMPANY}\Uninstall ${APPNAME}.lnk"
  RMDir "$SMPROGRAMS\${COMPANY}"

  DetailPrint "Removing desktop shortcut..."
  Delete "$DESKTOP\${APPNAME}.lnk"

  DetailPrint "Deleting application files..."
  Delete "$INSTDIR_VAR\ogamma-logger.exe"
  Delete "$INSTDIR_VAR\ha_node.exe"
  Delete "$INSTDIR_VAR\nssm.exe"
  Delete "$INSTDIR_VAR\vc_redist.x64.exe"
  Delete "$INSTDIR_VAR\Uninstall.exe"

  DetailPrint "Removing installation directory..."
  RMDir /r "$INSTDIR_VAR"

  ; Optional data removal with confirmation
  ${If} $UNINSTALL_REMOVE_DATA == 1
    ; If the key exists, prompt user and delete
    StrCmp $DATA_DIR_VAR "" no_data_dir has_data_dir

    has_data_dir:
      MessageBox MB_YESNO|MB_ICONQUESTION \
      "Do you want to remove the data folder at $DATA_DIR_VAR ?" \
      IDYES delete_data_dir IDNO keep_data_dir

    delete_data_dir:
	  DetailPrint "Removing data directory $DATA_DIR_VAR ..."
      RMDir /r "$DATA_DIR_VAR"
      Goto after_data_dir

    keep_data_dir:
      DetailPrint "User chose to keep data folder $DATA_DIR_VAR."
      Goto after_data_dir

    no_data_dir:
      DetailPrint "No data folder registry entry found."

    after_data_dir:
  ${Else}
    DetailPrint "Preserving data directory at $DATA_DIR_VAR"
  ${EndIf}
SectionEnd
